const { buildFileUri } = require('../config/s3.js');
const { formatDateTime, formatMoney } = require('../config/functionsGlobal.js');
const moment = require('moment-timezone');
const VentaModel = require('../models/Ventas.js');
const { default: mongoose } = require('mongoose');
const Usuarios = require('../models/Usuarios.js');

const getComisionesDespachos = async (req, res) => {
  const { distribuidor } = req.params;
  const { page = 1, fechaInicio, fechaFinal, typeUser = null } = req.query;

  const limit = 10;

  const options = {
    page,
    limit,
    sort: { fechaVenta: -1 },
    populate: [
      {
        path: 'producto',
        select: 'nombre imagen tipo_producto'
      },
      {
        path: 'usuario',
        select: 'nombre apellidoPaterno apellidoMaterno foto'
      },
      {
        path: 'despacho',
        select: 'nombre logo'
      },
      {
        path: 'who',
        select: 'nombre apellidoPaterno apellidoMaterno foto tipoUsuarios'
      }
    ],
    lean: true
  };

  try {
    let filtro;

    if (typeUser === 'promotor') {
      filtro = {
        'promotor.promotor': mongoose.Types.ObjectId(distribuidor)
      };
    } else if (typeUser === 'vendedor') {
      filtro = {
        'vendedor.vendedor': mongoose.Types.ObjectId(distribuidor)
      };
    } else {
      const usuariosReferidos = await Usuarios.find({
        'licencia.referidoPor': distribuidor
      }).select('_id');

      filtro = {
        $or: [
          { 'distribuidor.distribuidor': distribuidor },
          {
            'agente.agente': {
              $in: usuariosReferidos.map((usuario) => usuario._id)
            }
          }
        ]
      };
    }

    const query = filtro;

    if (fechaInicio && fechaFinal) {
      const fechaInicioUTC = moment(fechaInicio).startOf('day').toDate();
      const fechaFinalUTC = moment(fechaFinal).endOf('day').toDate();

      query.fechaVenta = {
        $gte: fechaInicioUTC,
        $lte: fechaFinalUTC
      };
    } else {
      return res.status(400).json({
        message: 'Favor de proporcionar una fecha de inicio y una fecha final'
      });
    }

    let importeField;
    if (typeUser === 'vendedor') {
      importeField = '$vendedor.importe';
    } else if (typeUser === 'promotor') {
      importeField = '$promotor.importe';
    } else {
      importeField = '$distribuidor.importe';
    }

    const [findVentas, totalComision] = await Promise.all([
      VentaModel.paginate(query, options),
      VentaModel.aggregate([
        { $match: query },
        {
          $group: {
            _id: null,
            totalComision: { $sum: importeField }
          }
        }
      ])
    ]);

    findVentas.docs = findVentas.docs.map((venta) => {
      const tipo = venta.despacho?._id ? 'Despacho' : 'Distribuidor';
      const { fechaVenta, producto, who } = venta;
      let imagen = '';
      let nameMostrar = '';

      if (tipo === 'Despacho') {
        imagen = venta.despacho.logo
          ? buildFileUri(
            `despachos/${venta?.despacho?._id}/logo/${venta.despacho.logo}`
          )
          : buildFileUri('default/icono_usuario_100x100_04.jpg');

        nameMostrar = venta.despacho.nombre;
      } else {
        imagen = venta.usuario.foto
          ? buildFileUri(`usuarios/${venta.usuario.foto}`)
          : buildFileUri('default/icono_usuario_100x100_04.jpg');
        nameMostrar = `${venta.usuario.nombre} ${venta.usuario.apellidoPaterno} ${venta.usuario.apellidoMaterno}`;
      }

      const tipoReferencia = venta.tipoReferencia;
      let tipoReferenciaLogo = '';

      if (tipoReferencia === 'Paypal') {
        tipoReferenciaLogo =
          'https://upload.wikimedia.org/wikipedia/commons/a/a4/Paypal_2014_logo.png';
      } else if (tipoReferencia === 'Stripe') {
        tipoReferenciaLogo =
          'https://s3-eu-west-1.amazonaws.com/tpd/logos/50489e6800006400051ae0d6/0x0.png';
      }

      let importeMostrar;
      if (typeUser === 'vendedor') {
        importeMostrar = formatMoney(venta.vendedor.importe);
      } else if (typeUser === 'promotor') {
        importeMostrar = formatMoney(venta.promotor.importe);
      } else {
        importeMostrar = formatMoney(venta.distribuidor.importe);
      }

      let comisionMostrar;
      if (typeUser === 'vendedor') {
        comisionMostrar = `${venta.vendedor.comision}%`;
      } else if (typeUser === 'promotor') {
        comisionMostrar = `${venta.promotor.comision}%`;
      } else {
        comisionMostrar = `${venta.distribuidor.comision}%`;
      }

      return {
        ...venta,
        fechaVenta: formatDateTime(fechaVenta),
        tipoReferenciaLogo,
        producto: {
          ...producto,
          imagen: producto.imagen
            ? buildFileUri(`productos/${producto.imagen}`)
            : ''
        },
        who: {
          ...who,
          foto: who?.foto
            ? buildFileUri(`usuarios/${who.foto}`)
            : buildFileUri('default/icono_usuario_100x100_04.jpg'),

          tipo: who?.tipoUsuarios?.distribuidor
            ? '(Distribuidor)'
            : who.tipoUsuarios.promotor
              ? '(Promotor)'
              : '(Vendedor)'
        },
        importe: formatMoney(venta.importe),
        distribuidor: {
          ...venta.distribuidor,
          importe: importeMostrar,
          comision: comisionMostrar
        },
        usuario: {
          ...venta.usuario,
          tipo,
          nombre: nameMostrar,
          imagen
        }
      };
    });

    return res.status(200).json({
      ventas: findVentas,
      totalComision: formatMoney(totalComision[0]?.totalComision || 0)
    });
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};

const getComisionesVendedoresPromotores = async (req, res) => {
  const { distribuidor } = req.params;
  const {
    page = 1,
    producto = '',
    fechaInicio = '',
    fechaFinal = '',
    typeUser = null
  } = req.query;

  const limit = 10;

  const options = {
    page,
    limit,
    sort: { fechaVenta: -1 },
    populate: [
      {
        path: 'producto',
        select: 'nombre imagen tipo_producto'
      },
      {
        path: 'usuario',
        select: 'nombre apellidoPaterno apellidoMaterno foto'
      },
      {
        path: 'despacho',
        select: 'nombre logo'
      },
      {
        path: 'who',
        select: 'nombre apellidoPaterno apellidoMaterno foto tipoUsuarios'
      }
    ],
    lean: true
  };

  try {
    let filtro;

    if (typeUser === 'promotor') {
      filtro = {
        'promotor.promotor': mongoose.Types.ObjectId(distribuidor)
      };
    }

    if (typeUser === 'vendedor') {
      filtro = {
        'vendedor.vendedor': mongoose.Types.ObjectId(distribuidor)
      };
    }

    const query = filtro;

    // Solo agrega el filtro de fecha si se proporcionan ambas fechas
    if (fechaInicio && fechaFinal) {
      const fechaInicioUTC = moment(fechaInicio).startOf('day').toDate();
      const fechaFinalUTC = moment(fechaFinal).endOf('day').toDate();

      query.fechaVenta = {
        $gte: fechaInicioUTC,
        $lte: fechaFinalUTC
      };
    }

    if (producto) {
      query.producto = producto;
    }

    let importeField;
    if (typeUser === 'vendedor') {
      importeField = '$vendedor.importe';
    }
    if (typeUser === 'promotor') {
      importeField = '$promotor.importe';
    }

    const [findVentas, totalComision] = await Promise.all([
      VentaModel.paginate(query, options),
      VentaModel.aggregate([
        { $match: query },
        {
          $group: {
            _id: null,
            totalComision: { $sum: importeField }
          }
        }
      ])
    ]);

    findVentas.docs = findVentas.docs.map((venta) => {
      const tipo = venta.despacho?._id ? 'Despacho' : 'Distribuidor';
      const { fechaVenta, producto, who } = venta;
      let imagen = '';
      let nameMostrar = '';

      if (tipo === 'Despacho') {
        imagen = venta.despacho.logo
          ? buildFileUri(
            `despachos/${venta?.despacho?._id}/logo/${venta.despacho.logo}`
          )
          : buildFileUri('default/icono_usuario_100x100_04.jpg');

        nameMostrar = venta.despacho.nombre;
      } else {
        imagen = venta.usuario.foto
          ? buildFileUri(`usuarios/${venta.usuario.foto}`)
          : buildFileUri('default/icono_usuario_100x100_04.jpg');
        nameMostrar = `${venta.usuario.nombre} ${venta.usuario.apellidoPaterno} ${venta.usuario.apellidoMaterno}`;
      }

      const tipoReferencia = venta.tipoReferencia;
      let tipoReferenciaLogo = '';

      if (tipoReferencia === 'Paypal') {
        tipoReferenciaLogo =
          'https://upload.wikimedia.org/wikipedia/commons/a/a4/Paypal_2014_logo.png';
      } else if (tipoReferencia === 'Stripe') {
        tipoReferenciaLogo =
          'https://s3-eu-west-1.amazonaws.com/tpd/logos/50489e6800006400051ae0d6/0x0.png';
      }

      let importeMostrar;
      if (typeUser === 'vendedor') {
        importeMostrar = formatMoney(venta.vendedor.importe);
      } else if (typeUser === 'promotor') {
        importeMostrar = formatMoney(venta.promotor.importe);
      } else {
        importeMostrar = formatMoney(venta.distribuidor.importe);
      }

      let comisionMostrar;
      if (typeUser === 'vendedor') {
        comisionMostrar = `${venta.vendedor.comision}%`;
      } else if (typeUser === 'promotor') {
        comisionMostrar = `${venta.promotor.comision}%`;
      } else {
        comisionMostrar = `${venta.distribuidor.comision}%`;
      }

      return {
        ...venta,
        fechaVenta: formatDateTime(fechaVenta),
        tipoReferenciaLogo,
        producto: {
          ...producto,
          imagen: producto.imagen
            ? buildFileUri(`productos/${producto.imagen}`)
            : ''
        },
        who: {
          ...who,
          foto: who?.foto
            ? buildFileUri(`usuarios/${who.foto}`)
            : buildFileUri('default/icono_usuario_100x100_04.jpg'),

          tipo: who?.tipoUsuarios?.distribuidor
            ? '(Distribuidor)'
            : who.tipoUsuarios.promotor
              ? '(Promotor)'
              : '(Vendedor)'
        },
        importe: formatMoney(venta.importe),
        distribuidor: {
          ...venta.distribuidor,
          importe: importeMostrar,
          comision: comisionMostrar
        },
        usuario: {
          ...venta.usuario,
          tipo,
          nombre: nameMostrar,
          imagen
        }
      };
    });

    return res.status(200).json({
      ventas: findVentas,
      totalComision: formatMoney(totalComision[0]?.totalComision || 0)
    });
  } catch (error) {
    console.error('Error:', error);
    return res
      .status(500)
      .json({ message: 'Error al obtener las ventas', error });
  }
};

module.exports = {
  getComisionesDespachos,
  getComisionesVendedoresPromotores
};
