const ClienteModel = require("../models/Clientes.js");
const {
  encriptar,
  desencriptar,
  generatePassword,
  sendWhatsappRecovery,
} = require("../config/functionsGlobal.js");
const { buildFileUri, uploadFile } = require("../config/s3.js");
const { RegistroUsuarioHTML } = require("../Mail/RegistroUsuarioHTML.js");
const { sendMail } = require("../config/mail.js");

const getClientes = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, search } = req.query;

  const options = {
    page,
    limit: 20,
    sort: { estatus: 1, nombre: 1 },
    populate: "estado",
  };

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  const query = {
    despacho,
  };

  if (estatus) {
    query.estatus = estatus;
  }

  if (search) {
    // query.nombre = { $regex: search, $options: 'i' };
    query.$or = [
      { nombre: { $regex: search, $options: "i" } },
      { rfc: { $regex: search, $options: "i" } },
      { correo: { $regex: search, $options: "i" } },
      { telefono: { $regex: search, $options: "i" } },
    ];
  }

  try {
    const findClientes = await ClienteModel.paginate(query, options);

    findClientes.docs = findClientes.docs.map((cliente) => {
      if (cliente?.foto) {
        cliente.foto = buildFileUri(`clientes/${cliente.foto}`);
      } else {
        cliente.foto = buildFileUri("default/icono_usuario_100x100_04.jpg");
      }

      if (cliente.password) {
        cliente.password = desencriptar(cliente.password);
      }

      return cliente;
    });

    res.status(200).json({ clientes: findClientes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const createCliente = async (req, res) => {
  const { despacho } = req.params;
  const { nombre, creadoPor } = req.body;

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  if (!nombre) {
    return res
      .status(400)
      .json({ message: "El nombre del cliente es requerido" });
  }

  if (!creadoPor) {
    return res
      .status(400)
      .json({ message: "El usuario que crea el cliente es requerido" });
  }

  const password = generatePassword();

  try {
    const objCliente = {
      despacho,
      creadoPor,
      password: encriptar(password),
      ...req.body,
    };

    const newCliente = await ClienteModel.create(objCliente);

    // enviar correo con la contraseña  const nombreCompleto = `${findCliente.nombre}`;
    const url = process.env.CLIENTE_APP || "https://clientes.nildda.com";
    const htmlRegistro = RegistroUsuarioHTML(
      nombre,
      req.body?.correo,
      password,
      url
    );

    if (req.body.telefono) {
      sendWhatsappRecovery({
        to: req.body.telefono,
        correo: req.body?.correo,
        password,
        url,
      });
    }

    if (req.body.correo)
      sendMail(htmlRegistro, "Recordatorio de contraseña", req.body.correo);

    res.status(201).json({
      message: "Cliente creado correctamente",
      cliente: newCliente,
    });
  } catch (error) {
    console.log(error);
    res.status(409).json({ message: error.message });
  }
};

const getCliente = async (req, res) => {
  const { id } = req.params;

  try {
    const cliente = await ClienteModel.findById(id);

    if (!cliente) {
      return res.status(404).json({ message: "Cliente no encontrado" });
    }

    res.status(200).json({ cliente });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const updateCliente = async (req, res) => {
  const { id } = req.params;
  const { nombre } = req.body;
  const foto = req?.files?.foto || null;

  if (!nombre) {
    return res
      .status(400)
      .json({ message: "El nombre del cliente es requerido" });
  }

  if (req.body.password) {
    req.body.password = encriptar(req.body.password);
  }

  try {
    const objCliente = {
      ...req.body,
    };

    if (foto) {
      const fotoUpdate = await uploadFile(foto, "clientes");

      if (fotoUpdate) {
        objCliente.foto = fotoUpdate.fileName;
      }
    }

    const updatedCliente = await ClienteModel.findByIdAndUpdate(
      id,
      objCliente,
      { new: true }
    );

    if (!updatedCliente) {
      return res.status(404).json({ message: "Cliente no encontrado" });
    }

    updatedCliente.password = desencriptar(updatedCliente.password);

    if (updatedCliente?.foto) {
      updatedCliente.foto = buildFileUri(`clientes/${updatedCliente.foto}`);
    } else {
      updatedCliente.foto = buildFileUri(
        "default/icono_usuario_100x100_04.jpg"
      );
    }

    res.status(200).json({
      message: "Cliente actualizado correctamente",
      cliente: updatedCliente,
    });
  } catch (error) {
    console.error(error);

    res.status(409).json({ message: error.message });
  }
};

const deleteCliente = async (req, res) => {
  const { id } = req.params;

  try {
    await ClienteModel.findByIdAndRemove(id);
    res.status(200).json({ message: "Cliente eliminado correctamente" });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getClientesSinPaginar = async (req, res) => {
  const { despacho } = req.params;

  if (!despacho) {
    return res.status(400).json({ message: "El despacho es requerido" });
  }

  try {
    const findClientes = await ClienteModel.find({
      despacho,
      estatus: "Activo",
    }).sort({ nombre: 1 });

    const clientes = findClientes.map((cliente) => {
      return {
        value: cliente._id,
        label: cliente.nombre,
        correo: cliente.correo,
        foto: cliente?.foto
          ? buildFileUri(`clientes/${cliente.foto}`)
          : buildFileUri("default/icono_usuario_100x100_04.jpg"),
      };
    });

    res.status(200).json({ clientes });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getClientes,
  createCliente,
  getCliente,
  updateCliente,
  deleteCliente,
  getClientesSinPaginar,
};
