const FavoritoModel = require('../models/BusquedaFavoritos.js');
const moment = require('moment');
const Chat = require('../models/ChatV2');
const getFavoritos = async (req, res) => {
  const { page = 1, search, tipo, orden } = req.query;
  const { despacho } = req.params;

  const options = {
    page,
    limit: 10,
    sort: orden ? { [orden]: 1 } : { tipo: 1, fechaCreacion: -1, nombre: 1 },
    populate: [
      {
        path: 'creadoPor',
        select: 'nombre apellidoPaterno apellidoMaterno foto'
      }
    ]
  };

  try {
    const query = {
      despacho
    };

    if (search) {
      query.nombre = { $regex: search, $options: 'i' };
    }

    if (tipo) {
      query.tipo = tipo;
    }

    console.log(options);
    const favoritos = await FavoritoModel.paginate(query, options);

    res.status(200).json(favoritos);
  } catch (error) {
    console.log(error);
    res.status(404).json({ message: error.message });
  }
};

const createFavorito = async (req, res) => {
  const { despacho } = req.params;
  const { id, nombre, creadoPor, epoca, tipo } = req.body;

  if (!nombre) {
    return res.status(400).json({ message: 'El nombre es requerido' });
  }

  if (!id) {
    return res.status(400).json({ message: 'El ID  es requerido' });
  }

  if (!creadoPor) {
    return res.status(400).json({ message: 'El usuario es requerido' });
  }

  const materia = {
    nombre,
    despacho,
    id,
    fechaCreacion: moment(),
    creadoPor,
    epoca: epoca || null,
    tipo
  };

  try {
    // contar si ya existe la tesis

    const tesis = await FavoritoModel.countDocuments({ id, despacho, tipo });

    if (tesis) {
      return res
        .status(409)
        .json({ message: `la ${tipo} ya existe en favoritos` });
    }

    const newTesis = FavoritoModel.create(materia);
    res
      .status(201)
      .json({ tesis: newTesis, message: `${tipo} agregada a favoritos` });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const createOrDeleteFavoritoChat = async (req, res) => {
  const { id, creadoPor, despacho, nombre, texto } = req.body;

  if (!id) {
    return res.status(400).json({ message: 'El ID es requerido' });
  }

  if (!creadoPor) {
    return res.status(400).json({ message: 'El usuario es requerido' });
  }

  try {
    const findFavorite = await FavoritoModel.findOne({ id, despacho });

    if (findFavorite) {
      await FavoritoModel.findOneAndDelete({ id, despacho });

      await Chat.findOneAndUpdate(
        { _id: id },
        { favorito: null },
        { new: true }
      );

      return res.status(200).json({ message: 'Favorito eliminado' });
    }

    const newFavorite = await FavoritoModel.create({
      despacho,
      id,
      tipo: 'Chat GPT',
      nombre,
      creadoPor,
      texto
    });

    await Chat.findOneAndUpdate(
      { _id: id },
      { favorito: newFavorite._id },
      { new: true }
    );

    res.status(201).json({ message: 'Favorito creado', favorito: newFavorite });
  } catch (error) {
    res.status(409).json({ message: error.message });
  }
};

const getById = async (req, res) => {
  const { despacho, tipo, id } = req.params;

  try {
    const tesis = await FavoritoModel.findOne({ id, despacho, tipo });
    if (!tesis) {
      return res.status(404).json({ message: 'No se encontró la tesis' });
    }

    res.status(200).json(tesis);
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

const deleteFavorito = async (req, res) => {
  const { despacho, tipo, id } = req.params;

  try {
    const tesis = await FavoritoModel.findOneAndDelete({ id, tipo, despacho });
    if (!tesis) {
      return res.status(404).json({ message: `No se encontró la ${tipo}` });
    }

    res.status(200).json({ message: `La ${tipo} ha sido eliminada de favoritos` });
  } catch (error) {
    res.status(404).json({ message: error.message });
  }
};

module.exports = {
  getFavoritos,
  createFavorito,
  getById,
  deleteFavorito,
  createOrDeleteFavoritoChat
};
