/* eslint-disable prefer-promise-reject-errors */
const { exec } = require('child_process');
const fs = require('fs');
const path = require('path');
const { uploadFile } = require('../config/s3.js'); // Ensure this is correctly implemented

const runCommand = (command) => {
  return new Promise((resolve, reject) => {
    exec(command, (error, stdout, stderr) => {
      if (error) {
        return reject(`Error: ${error.message}`);
      }

      // Log non-critical stderr output and continue if no error was reported
      if (stderr) {
        console.warn(`Stderr: ${stderr}`); // Logging for information only
      }

      resolve(stdout);
    });
  });
};

const backupController = async (req, res) => {
  try {
    const __dirname = path.resolve();
    const backupDir = path.join(__dirname, 'backup');

    // Verify if the backup directory exists, create it if not
    if (!fs.existsSync(backupDir)) {
      fs.mkdirSync(backupDir, { recursive: true });
    }

    const uri = process.env.MONGODB_URI;
    if (!uri) {
      throw new Error('La URI de MongoDB no está configurada en las variables de entorno.');
    }

    const fechaDia = new Date().toISOString();
    const backupName = `backup-${fechaDia}.gzip`;
    const dumpPath = path.join(backupDir, backupName);

    // Command to execute the backup
    const command = `mongodump --uri '${uri}' --gzip --archive=${dumpPath}`;

    // Execute the mongodump command
    await runCommand(command);

    // Verify if the file was created correctly
    if (!fs.existsSync(dumpPath)) {
      throw new Error(`Archivo de backup no encontrado en ${dumpPath}`);
    }

    // Read the file and upload it to S3
    fs.createReadStream(dumpPath);

    // Define the MIME type if necessary
    // const fileType = 'application/gzip'; // Define the MIME type of the file

    // Upload the file to S3
    const file = {
      name: backupName,
      tempFilePath: dumpPath,
      mimetype: 'application/gzip'
    };

    uploadFile(file, 'backups');

    // Clean up the local file after upload
    // fs.unlinkSync(dumpPath);

    // Success response
    res.json({ message: 'Backup realizado con éxito y subido a S3' });
  } catch (error) {
    console.error('Error al realizar el backup:', error);
    res.status(500).json({ error: 'Error al realizar el backup', message: error.message || error });
  }
};

module.exports = {
  backupController
};
