/* eslint-disable array-callback-return */
const { buildFileUri } = require('../config/s3.js');
const Alertas = require('../models/Alertas.js');
const moment = require('moment-timezone');

const createAlertas = async (req, res) => {
  try {
    const { despacho } = req.params;
    const { creadoPor, dirigidos, titulo, mensaje, color } = req.body;

    if (!despacho) {
      return res.status(400).json({ message: 'El despacho es requerido' });
    }

    if (!creadoPor) {
      return res.status(400).json({ message: 'El usuario que crea la alerta es requerido' });
    }

    if (!dirigidos) {
      return res.status(400).json({ message: 'Los usuarios a los que se dirige la alerta son requeridos' });
    }

    if (!titulo) {
      return res.status(400).json({ message: 'El título de la alerta es requerido' });
    }

    if (!mensaje) {
      return res.status(400).json({ message: 'El mensaje de la alerta es requerido' });
    }

    const objAlerta = {
      creadoPor,
      dirigidos: dirigidos.map(usuario => {
        return {
          estatus: 'Pendiente',
          usuarios: usuario
        };
      }),
      titulo,
      mensaje,
      fecha: moment().tz('America/Mexico_City'),
      despacho,
      color
    };

    const newAlerta = await Alertas.create(objAlerta);

    res.status(201).json({
      message: 'Alerta creada exitosamente.',
      asunto: newAlerta
    });
  } catch (error) {
    res.status(404).json({
      message: error.message
    });
  }
};

const getAlertas = async (req, res) => {
  const { despacho } = req.params;
  const { estatus, page = 1, usuario, search, tipo = 'All' } = req.query;

  const options = {
    page,
    limit: 10,
    sort: {
      fecha: -1,
      'dirigidos.estatus': -1
    },
    populate: [
      {
        path: 'dirigidos.usuarios',
        select: 'nombre apellidoPaterno apellidoMaterno foto'
      },
      {
        path: 'creadoPor',
        select: 'nombre apellidoPaterno apellidoMaterno foto'
      }
    ]
  };

  if (!despacho) {
    return res.status(400).json({ message: 'El despacho es requerido' });
  }

  if (!usuario) {
    return res.status(400).json({ message: 'El usuario es requerido' });
  }

  const query = {
    despacho
  };

  if (tipo === 'All') {
    query.$or = [
      {
        'dirigidos.usuarios': { $in: [usuario] }
      },
      {
        creadoPor: usuario
      }
    ];
  }

  if (tipo === 'ForMe') {
    query['dirigidos.usuarios'] = { $in: [usuario] };
  }

  if (tipo === 'CreatedByMe') {
    query.creadoPor = usuario;
  }

  if (search) {
    query.titulo = { $regex: search, $options: 'i' };
  }

  if (estatus) {
    query['dirigidos.estatus'] = estatus;
  }

  try {
    const findAlertas = await Alertas.paginate(query, options);

    findAlertas.docs.forEach(alerta => {
      alerta.dirigidos.forEach(dirigido => {
        const foto = dirigido.usuarios.foto;
        if (foto && !foto.startsWith('http')) {
          dirigido.usuarios.foto = buildFileUri(`usuarios/${foto}`);
        }
      });

      const foto = alerta.creadoPor.foto;

      if (foto && !foto.startsWith('http')) {
        alerta.creadoPor.foto = buildFileUri(`usuarios/${foto}`);
      }
    });

    // ordenar por nombre los usuarios dirigidos
    findAlertas.docs.forEach(alerta => {
      alerta.dirigidos.sort((a, b) => {
        if (a.usuarios.nombre < b.usuarios.nombre) {
          return -1;
        }
        if (a.usuarios.nombre > b.usuarios.nombre) {
          return 1;
        }
        return 0;
      });
    });

    const pendientes = await countAlertas(usuario);

    return res.status(200).json({ alertas: findAlertas, pendientes });
  } catch (error) {
    return res.status(404).json({ message: error.message });
  }
};

const countAlertas = async (usuario) => {
  const query = {
    dirigidos: {
      $elemMatch: {
        usuarios: usuario,
        estatus: 'Pendiente'
      }
    }
  };

  try {
    const count = await Alertas.countDocuments(query);
    console.log(count);
    return count;
  } catch (error) {
    console.error(error);
    return 0;
  }
};

const getAlertaById = async (req, res) => {
  const { id } = req.params;

  try {
    const alerta = await Alertas.findById(id).populate([{
      path: 'dirigidos.usuarios',
      select: 'nombre apellidoPaterno apellidoMaterno foto'
    }]);

    if (!alerta) {
      return res.status(404).json({ message: 'La alerta no existe' });
    }

    return res.status(200).json({ alerta });
  } catch (error) {
    return res.status(404).json({ message: error.message });
  }
};

const updateAlerta = async (req, res) => {
  const { id } = req.params;
  const { dirigidos, titulo, mensaje, color } = req.body;

  try {
    const alerta = await Alertas.findById(id);

    if (!alerta) {
      return res.status(404).json({ message: 'La alerta no existe' });
    }

    // eliminar los dirigidos que no están, y si hay nuevo colocarlos con estatus pendientes
    const nuevosDirigidos = dirigidos.filter(dirigido => !alerta.dirigidos.some(d => d.usuarios.toString() === dirigido));

    alerta.dirigidos = alerta.dirigidos.filter(dirigido => dirigidos.includes(dirigido.usuarios.toString()));

    nuevosDirigidos.forEach(nuevoDirigido => {
      alerta.dirigidos.push({
        estatus: 'Pendiente',
        usuarios: nuevoDirigido
      });
    });

    alerta.titulo = titulo;
    alerta.mensaje = mensaje;
    alerta.color = color;

    await alerta.save();

    return res.status(200).json({ message: 'Estatus de la alerta actualizado' });
  } catch (error) {
    return res.status(404).json({ message: error.message });
  }
};

const updateAlertaEstatus = async (req, res) => {
  const { id } = req.params;
  const { estatus, usuario, comentario } = req.body;

  try {
    const alerta = await Alertas.findById(id);

    if (!alerta) {
      return res.status(404).json({ message: 'La alerta no existe' });
    }

    const dirigido = alerta.dirigidos.find(d => d.usuarios.toString() === usuario);

    if (!dirigido) {
      return res.status(404).json({ message: 'El usuario no está en la alerta' });
    }

    dirigido.estatus = estatus;
    if (comentario) {
      dirigido.comentario = comentario;
    }

    await alerta.save();

    return res.status(200).json({ message: 'Estatus de la alerta actualizado' });
  } catch (error) {
    return res.status(404).json({ message: error.message });
  }
};

const deleteAlerta = async (req, res) => {
  const { id } = req.params;

  try {
    const alerta = await Alertas.findById(id);

    if (!alerta) {
      return res.status(404).json({ message: 'La alerta no existe' });
    }

    await alerta.remove();

    return res.status(200).json({ message: 'Alerta eliminada' });
  } catch (error) {
    return res.status(404).json({ message: error.message });
  }
};

module.exports = {
  createAlertas,
  getAlertas,
  countAlertas,
  updateAlerta,
  getAlertaById,
  deleteAlerta,
  updateAlertaEstatus
};
