const multer = require("multer");
const path = require("path");
const { mkdir } = require("fs");
const { promisify } = require("util");

const mkdirAsync = promisify(mkdir);

const generateFilename = (originalname) => {
  const uniqueSuffix = "-" + Date.now() + "-" + Math.round(Math.random() * 1e9);
  const extname = path.extname(originalname);
  const basename = path.basename(originalname, extname);
  return basename + uniqueSuffix + extname;
};

const configureMulterSingle = (fieldName = "image", destinationFolder = "") => {
  const storage = multer.diskStorage({
    destination: async (req, file, cb) => {
      const folderPath = path.join("src/uploads", destinationFolder);
      try {
        await mkdirAsync(folderPath, { recursive: true });
        cb(null, folderPath);
      } catch (err) {
        cb(err);
      }
    },
    filename: (req, file, cb) => {
      cb(null, generateFilename(file.originalname));
    },
  });

  return multer({ storage }).single(fieldName);
};

const configureMulterArray = (
  fieldName = "documents",
  destinationFolder = ""
) => {
  const storage = multer.diskStorage({
    destination: async (req, file, cb) => {
      const folderPath = path.join("src/uploads", destinationFolder);
      try {
        await mkdirAsync(folderPath, { recursive: true });
        cb(null, folderPath);
      } catch (err) {
        cb(err);
      }
    },
    filename: (req, file, cb) => {
      cb(null, generateFilename(file.originalname));
    },
  });

  return multer({ storage }).array(fieldName, 30);
};

const configureMulterMixed = (destinationFolder = "", names = []) => {
  const storage = multer.diskStorage({
    destination: async (req, file, cb) => {
      const folderPath = path.join("src/uploads", destinationFolder);
      try {
        await mkdirAsync(folderPath, { recursive: true });
        cb(null, folderPath);
      } catch (err) {
        cb(err);
      }
    },
    filename: (req, file, cb) => {
      cb(null, generateFilename(file.originalname));
    },
  });

  return multer({
    storage,
  }).fields(names.map((name) => ({ name, maxCount: 1 })));
};

module.exports = {
  configureMulterSingle,
  configureMulterArray,
  configureMulterMixed,
};
